# 機能設計書 80-型格子（Type Lattice）

## 概要

本ドキュメントは、Juliaコンパイラの型推論で使用する格子（lattice）構造の設計を記述する。`Conditional`、`MustAlias`、`PartialStruct`、`LimitedAccuracy` 等の拡張格子要素と、それらの間の半順序関係（`⊑`）、合流演算（`tmerge`）、交差演算（`tmeet`）を定義する。

### 本機能の処理概要

型格子機能は、Juliaの型推論（抽象解釈）において、Julia型システム（`Type`）のみでは表現できない精密な型情報を格子要素として表現・操作するための基盤を提供する。格子要素は階層的に構成され、各層が特定の拡張情報（定数値、部分的構造体情報、条件分岐型精密化、フィールドエイリアス追跡、近似精度情報）を付加する。

**業務上の目的・背景**: Juliaの型推論は抽象解釈に基づいており、正確な推論結果を得るためには、ネイティブの型システムを超えた精密な格子構造が必要である。例えば、`isa(x, Int)` の結果が `true` のブランチでは `x` の型を `Int` に精密化できるが、これは `Bool` 型だけでは表現できず、`Conditional` 格子要素が必要となる。

**機能の利用シーン**: コンパイラの型推論（抽象解釈）フェーズにおいて、変数の型状態管理、条件分岐の型精密化、定数伝播、構造体フィールドの部分的型情報追跡、再帰呼び出しでの近似精度追跡に使用される。

**主要な処理内容**:
1. **格子要素の定義**: `Conditional`、`MustAlias`、`InterConditional`、`InterMustAlias`、`PartialTypeVar`、`LimitedAccuracy`、`StateUpdate`、`StateRefinement`、`NotFound` の構造体定義
2. **半順序関係 `⊑`**: 各格子層に対応した部分順序の定義（`ConstsLattice`、`PartialsLattice`、`ConditionalsLattice`、`MustAliasesLattice`、`InferenceLattice`）
3. **合流演算 `tmerge`**: 二つの格子要素の上界（join）を計算する演算。複雑度制限付き
4. **交差演算 `tmeet`**: 格子要素と型の交差（meet）を計算する演算
5. **型複雑度制限**: `limit_type_size` による型の複雑度制限と `type_more_complex` による複雑度比較
6. **状態管理**: `smerge`、`stupdate!`、`stoverwrite!` による変数テーブルの管理

**関連システム・外部連携**: `Compiler/src/abstractinterpretation.jl`（抽象解釈エンジン）、`Compiler/src/types.jl`（`AbstractLattice` 抽象型の定義）、Core モジュール（`Const`、`InterConditional`、`PartialStruct` はキャッシュ互換のため Core で定義）

**権限による制御**: 特別な権限制御はない。コンパイラ内部のみで使用される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | コンパイラ内部処理（ユーザ画面なし） |

## 機能種別

コンパイラ内部データ構造・アルゴリズム

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| a | Any（格子要素） | Yes | 比較・演算の左辺格子要素 | assert_nested_slotwrapper で入れ子検証 |
| b | Any（格子要素） | Yes | 比較・演算の右辺格子要素 | assert_nested_slotwrapper で入れ子検証 |
| lattice | AbstractLattice | Yes | 使用する格子の種類 | 格子層の型で制約 |
| t | Type | Yes (limit_type_size) | 制限対象の型 | isa(t, Type) |
| compare | Type | Yes (limit_type_size) | 比較基準の型 | isa(compare, Type) |
| source | Type | Yes (limit_type_size) | 許容される型値の出典 | isa(source, Type) |

### 入力データソース

- 抽象解釈エンジンからの型推論結果
- 各 SSA 値・スロット変数の型状態

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result (⊑) | Bool | 半順序関係の判定結果 |
| result (tmerge) | Any（格子要素） | 合流演算の結果（上界） |
| result (tmeet) | Any（格子要素） | 交差演算の結果（下界） |
| result (limit_type_size) | Type | 複雑度が制限された型 |
| result (widenconst) | Type | 格子要素からネイティブ型への変換結果 |

### 出力先

コンパイラの抽象解釈エンジン（`abstractinterpretation.jl`）へのフィードバック

## 処理フロー

### 処理シーケンス

```
1. 格子要素の生成
   ├─ Conditional(slot, ssadef, thentype, elsetype): 条件分岐の型精密化情報
   ├─ MustAlias(slot, ssadef, vartyp, fldidx, fldtyp): フィールドエイリアス追跡
   ├─ LimitedAccuracy(typ, causes): 再帰検出時の近似精度追跡
   └─ PartialStruct(lattice, typ, undefs, fields): 構造体フィールドの部分型情報

2. 半順序関係の判定 (⊑)
   └─ 格子層ごとにディスパッチ:
      InferenceLattice → (LimitedAccuracy処理)
      AnyConditionalsLattice → (Conditional処理)
      AnyMustAliasesLattice → (MustAlias処理)
      PartialsLattice → (PartialStruct/PartialOpaque処理)
      ConstsLattice → (Const/PartialTypeVar処理)
      JLTypeLattice → (Type <: 判定)

3. 合流演算 (tmerge)
   └─ 各格子層で固有のマージロジック:
      InferenceLattice → tmerge_limited (LimitedAccuracy + causes追跡)
      ConditionalsLattice → Conditional間のマージ or Bool化
      AnyMustAliasesLattice → widenmustalias後にマージ
      PartialsLattice → tmerge_partial_struct or PartialOpaqueマージ
      ConstsLattice → widenlattice後にマージ
      JLTypeLattice → tmerge_types_slow (Union簡約化)

4. 型複雑度制限 (limit_type_size)
   ├─ type_more_complex で複雑度を比較
   ├─ _limit_type_size で型パラメータを再帰的に簡約化
   └─ is_derived_type_from_any で既知の型かチェック
```

### フローチャート

```mermaid
flowchart TD
    A[格子要素 a, b] --> B{演算の種類}
    B -->|"⊑ (半順序)"| C[格子層ディスパッチ]
    B -->|"tmerge (合流)"| D[格子層ディスパッチ]
    B -->|"tmeet (交差)"| E[格子層ディスパッチ]

    C --> C1{InferenceLattice}
    C1 -->|LimitedAccuracy| C2[ignorelimited後に比較]
    C1 -->|else| C3{ConditionalsLattice}
    C3 -->|Conditional| C4[issubconditional]
    C3 -->|else| C5{PartialsLattice}
    C5 -->|PartialStruct| C6[フィールドごとに⊑判定]
    C5 -->|else| C7{ConstsLattice}
    C7 -->|Const| C8[val === 比較]
    C7 -->|else| C9[JLTypeLattice: a <: b]

    D --> D1{tmerge_fast_path}
    D1 -->|fast path hit| D2[結果返却]
    D1 -->|miss| D3[格子層固有マージ]
    D3 --> D4[tmerge_types_slow]
    D4 --> D5[Union簡約化]
    D5 --> D6[issimpleenoughtype判定]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-80-01 | 入れ子禁止 | Conditional/MustAlias 内に別の Conditional/MustAlias を入れ子にできない | 全格子要素生成時（assert_nested_slotwrapper で検証） |
| BR-80-02 | LimitedAccuracy入れ子禁止 | LimitedAccuracy 内に別の LimitedAccuracy を入れ子にできない | LimitedAccuracy 生成時（コンストラクタで @assert） |
| BR-80-03 | Conditional→Bool変換 | thentype===Bottom なら Const(false)、elsetype===Bottom なら Const(true)、それ以外は Bool | widenconditional |
| BR-80-04 | MustAlias→fldtyp変換 | widenmustalias は fldtyp を返す | widenmustalias |
| BR-80-05 | 複雑度上限 | MAX_TYPEUNION_COMPLEXITY=3、MAX_TYPEUNION_LENGTH=3 | issimpleenoughtype |
| BR-80-06 | tmerge結合性 | tmerge は厳密な最小上界ではなく、複雑度制限付きの近似上界を返す | 全 tmerge 呼び出し |
| BR-80-07 | LimitedAccuracy causes追跡 | LimitedAccuracy のマージ時に causes を union/merge して近似の原因を追跡する | tmerge_limited |
| BR-80-08 | PartialStruct互換性 | PartialStruct の生成時に undefs（未定義フィールド情報）を追跡する | PartialStruct コンストラクタ |
| BR-80-09 | Inter変換 | ローカル推論の Conditional/MustAlias はプロシージャ間では InterConditional/InterMustAlias に変換される | widenreturn |
| BR-80-10 | widenlattice委譲 | 各格子層は自分が処理できない要素を widenlattice で下位層に委譲する | 全格子演算 |

### 計算ロジック

- `⊑(InferenceLattice, a, b)`: `ignorelimited(a) ⊑ ignorelimited(b)` が成り立ち、`b` が LimitedAccuracy の場合は `b.causes ⊆ a.causes` も要求する（typelattice.jl 395-412行目）
- `tmerge(JLTypeLattice, a, b)`: 具象型同士なら `Union{a,b}`、それ以外は `tmerge_types_slow` で共通の TypeName を見つけて簡約化（typelimits.jl 736-893行目）
- `limit_type_size(t, compare, source, depth, len)`: `type_more_complex(t, compare, ...)` が true の場合のみ `_limit_type_size` で再帰的に型パラメータを簡約化（typelimits.jl 17-33行目）

## データベース操作仕様

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| AssertionError | アサーション | Conditional/MustAlias の入れ子検出 | assert_nested_slotwrapper が検出。格子要素の生成ロジックの修正が必要 |
| AssertionError | アサーション | LimitedAccuracy の入れ子検出 | LimitedAccuracy コンストラクタの @assert。生成元の修正が必要 |
| ErrorException | 例外 | widenconst に TypeVar/Vararg/LimitedAccuracy が渡された | 呼び出し側で事前に widen すること |
| ErrorException | 例外 | widenlattice(JLTypeLattice()) の呼び出し | JLTypeLattice は最下層であり、さらに下位の格子は存在しない |

### リトライ仕様

リトライ不要（コンパイル時の純粋計算）

## トランザクション仕様

該当なし

## パフォーマンス要件

- `tmerge_fast_path` により、`Union{}` や同一要素、部分順序関係にある要素のマージを高速に処理する
- `issimpleenoughtype` により型の Union 長さ・複雑度を `MAX_TYPEUNION_LENGTH=3` / `MAX_TYPEUNION_COMPLEXITY=3` に制限し、型推論の停止性を保証する
- `@nospecializeinfer` アノテーションにより、格子演算関数の不要な特殊化を抑制してコンパイル時間を削減する
- `pointer_from_objref` による具象型のポインタ比較で高速な等価判定を実現（typelimits.jl 742行目）

## セキュリティ考慮事項

- コンパイラ内部のみで使用され、外部からの直接アクセスはない
- 型格子の計算結果が不正確な場合でも、生成されるコードの安全性には影響しない（型推論は最適化の精度に影響するが、安全性は型チェックで保証される）

## 備考

- `Const`、`InterConditional`、`PartialStruct` は Core モジュールで定義されており、グローバルコードキャッシュ内で使用可能
- `Conditional` と `InterConditional` はそれぞれローカル推論とプロシージャ間呼び出しで使用され、互いにマージされることはない
- `MustAlias` は現在 abstractinterpretation でのみ使用され、最適化フェーズでは使用されない（typelattice.jl 92行目のコメント）
- 格子層の標準的な積み重ね順序は: `JLTypeLattice` → `ConstsLattice` → `PartialsLattice` → `ConditionalsLattice` → `MustAliasesLattice` → `InferenceLattice`
- `fallback_lattice` は `InferenceLattice(BaseInferenceLattice.instance)` として定義される（abstractlattice.jl 301行目）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 格子の階層構造を理解する

格子層の定義と `widenlattice` による委譲チェーンを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | types.jl | `Compiler/src/types.jl` | `abstract type AbstractLattice end`（38行目）: 格子の抽象基底型 |
| 1-2 | abstractlattice.jl | `Compiler/src/abstractlattice.jl` | JLTypeLattice（7-13行目）: 最下層、Julia型のみ。widenlattice はエラー |
| 1-3 | abstractlattice.jl | `Compiler/src/abstractlattice.jl` | ConstsLattice（16-22行目）: Const + PartialTypeVar を追加 |
| 1-4 | abstractlattice.jl | `Compiler/src/abstractlattice.jl` | PartialsLattice（25-33行目）: PartialStruct + PartialOpaque を追加 |
| 1-5 | abstractlattice.jl | `Compiler/src/abstractlattice.jl` | ConditionalsLattice（36-44行目）: Conditional を追加 |
| 1-6 | abstractlattice.jl | `Compiler/src/abstractlattice.jl` | MustAliasesLattice（58-66行目）: MustAlias を追加 |
| 1-7 | abstractlattice.jl | `Compiler/src/abstractlattice.jl` | InferenceLattice（87-96行目）: LimitedAccuracy を追加（最上層） |
| 1-8 | abstractlattice.jl | `Compiler/src/abstractlattice.jl` | fallback_lattice定義（301-302行目）: 標準的な格子インスタンス |

**読解のコツ**: 各格子層は `widenlattice` により下位層への参照を持ち、自分が処理できない要素は `widenlattice` で委譲する。`is_valid_lattice_norec` が各層の担当する要素の判定に使用される。格子の積み重ね順序（下から上）は: `JLTypeLattice` → `ConstsLattice` → `PartialsLattice` → `ConditionalsLattice` → `MustAliasesLattice` → `InferenceLattice`。

#### Step 2: 格子要素の構造体を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | typelattice.jl | `Compiler/src/typelattice.jl` | Conditional（44-63行目）: slot/ssadef/thentype/elsetype/isdefined を持つ。条件分岐後の型精密化に使用 |
| 2-2 | typelattice.jl | `Compiler/src/typelattice.jl` | MustAlias（94-111行目）: slot/ssadef/vartyp/fldidx/fldtyp を持つ。オブジェクトフィールドの同一性追跡 |
| 2-3 | typelattice.jl | `Compiler/src/typelattice.jl` | InterConditional/InterMustAlias（66-69, 118-134行目）: プロシージャ間版。ssadef を持たない |
| 2-4 | typelattice.jl | `Compiler/src/typelattice.jl` | PartialTypeVar（142-149行目）: TypeVar の境界の確定度を追跡 |
| 2-5 | typelattice.jl | `Compiler/src/typelattice.jl` | LimitedAccuracy（208-216行目）: 再帰検出時の近似を表す。typ + causes（InferenceState の集合）を保持 |
| 2-6 | typelattice.jl | `Compiler/src/typelattice.jl` | NotFound/NOT_FOUND（218-229行目）: SSA値の未解析状態を表すシングルトン |
| 2-7 | typelattice.jl | `Compiler/src/typelattice.jl` | StateUpdate/StateRefinement（151-168行目）: 変数テーブルの更新・精密化操作 |

**読解のコツ**: `Conditional` と `MustAlias` のコンストラクタは `may_form_limited_typ` を呼び出し、引数に `LimitedAccuracy` が含まれる場合は自動的に `LimitedAccuracy` でラップして返す。また、`assert_nested_slotwrapper` により入れ子が禁止されている。

#### Step 3: 半順序関係（⊑）を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | abstractlattice.jl | `Compiler/src/abstractlattice.jl` | ⊑ の定義と ⊏、⋤、is_lattice_equal（144-181行目）: 基本インターフェース |
| 3-2 | typelattice.jl | `Compiler/src/typelattice.jl` | ⊑(ConstsLattice, a, b)（532-552行目）: Const 同士は val === で比較、Const ⊑ Type は isa で判定 |
| 3-3 | typelattice.jl | `Compiler/src/typelattice.jl` | ⊑(PartialsLattice, a, b)（454-530行目）: PartialStruct はフィールドごとに再帰的に ⊑ 判定 |
| 3-4 | typelattice.jl | `Compiler/src/typelattice.jl` | ⊑(AnyConditionalsLattice, a, b)（414-439行目）: Conditional 同士は issubconditional、fast path あり |
| 3-5 | typelattice.jl | `Compiler/src/typelattice.jl` | ⊑(InferenceLattice, a, b)（395-412行目）: LimitedAccuracy は ignorelimited 後に比較し、causes の包含関係も確認 |

**主要処理フロー**:
1. **395-412行目**: InferenceLattice 層で LimitedAccuracy を剥がし、下位層に委譲。b が LimitedAccuracy の場合、`b.causes ⊆ a.causes` を追加で要求
2. **414-439行目**: Conditional 同士の比較は `issubconditional`（slot, ssadef 一致 + thentype/elsetype の ⊑）
3. **454-530行目**: PartialStruct ⊑ PartialStruct はフィールド数一致 + 各フィールドの ⊑。Const ⊑ PartialStruct は各 getfield 値との ⊑

#### Step 4: 合流演算（tmerge）を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | typelimits.jl | `Compiler/src/typelimits.jl` | tmerge_fast_path（403-415行目）: Union{} / 同一 / 部分順序のファストパス |
| 4-2 | typelimits.jl | `Compiler/src/typelimits.jl` | tmerge(ConditionalsLattice)（499-537行目）: 同じ slot の Conditional 同士は thentype/elsetype をマージ |
| 4-3 | typelimits.jl | `Compiler/src/typelimits.jl` | tmerge_partial_struct（593-676行目）: PartialStruct 同士のフィールドごとマージ |
| 4-4 | typelimits.jl | `Compiler/src/typelimits.jl` | tmerge(JLTypeLattice)（736-750行目）: 具象型なら Union、それ以外は tmerge_types_slow |
| 4-5 | typelimits.jl | `Compiler/src/typelimits.jl` | tmerge_types_slow（779-893行目）: Union の簡約化。同名 TypeName は wrapper に拡大、Tuple は tuplemerge |
| 4-6 | typelimits.jl | `Compiler/src/typelimits.jl` | tmerge_limited（440-489行目）: LimitedAccuracy の causes 追跡付きマージ |

**主要処理フロー**:
1. **403-415行目**: `tmerge_fast_path` で `Union{}` / 同一 / 一方が他方の部分型の場合を高速処理
2. **499-527行目**: 同一 slot の Conditional 同士 → thentype/elsetype を各々 tmerge → 新 Conditional。異なる slot → Bool に退化
3. **779-893行目**: `tmerge_types_slow` で Union の各要素を DataType に分解し、同一 TypeName のペアを wrapper に拡大。`issimpleenoughtype` で十分単純かチェック

#### Step 5: 型複雑度制限を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | typelimits.jl | `Compiler/src/typelimits.jl` | MAX_TYPEUNION_COMPLEXITY/LENGTH（7-8行目）: 定数 3/3 |
| 5-2 | typelimits.jl | `Compiler/src/typelimits.jl` | limit_type_size（17-33行目）: type_more_complex 判定 → _limit_type_size で簡約 |
| 5-3 | typelimits.jl | `Compiler/src/typelimits.jl` | type_more_complex（223-307行目）: 再帰的な型複雑度比較 |
| 5-4 | typelimits.jl | `Compiler/src/typelimits.jl` | _limit_type_size（86-189行目）: 型パラメータの再帰的簡約。Tuple は要素ごとに制限 |
| 5-5 | typelimits.jl | `Compiler/src/typelimits.jl` | is_derived_type（37-75行目）: 型が comparison の部分型パラメータに由来するか検査 |

**主要処理フロー**:
1. **17-33行目**: `limit_type_size` は `type_more_complex` が true の場合のみ簡約を実施。結果が `t <: r` を満たさない場合は `Any` まで拡大
2. **86-189行目**: `_limit_type_size` は Tuple の場合は要素ごとに `__limit_type_size` を適用、DataType の場合は `name.wrapper` に拡大
3. **223-307行目**: `type_more_complex` は Union/UnionAll/DataType/TypeVar の構造を再帰的に比較し、`is_derived_type_from_any` で既知の型なら false を返す

#### Step 6: 状態管理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 6-1 | typelattice.jl | `Compiler/src/typelattice.jl` | smerge（728-733行目）: VarState のマージ。tmerge + undef フラグの or 結合 |
| 6-2 | typelattice.jl | `Compiler/src/typelattice.jl` | schanged（735-736行目）: VarState の変化検出 |
| 6-3 | typelattice.jl | `Compiler/src/typelattice.jl` | stupdate!（738-754行目）: VarTable 全体の更新。パス収束基準 (path-convergence criterion) による SSA 構築 |
| 6-4 | typelattice.jl | `Compiler/src/typelattice.jl` | stoverwrite1!（763-770行目）: 単一スロットの上書き。ssadef 変更により古い Conditional/MustAlias が自動的に失効 |

**読解のコツ**: `stupdate!` のコメント（743-747行目）にあるように、`join_pc` は基本ブロック先頭の仮想的な phi ブロックの PC を表し、SSA 構築のパス収束基準（path-convergence criterion）を実装している。

### プログラム呼び出し階層図

```
格子演算の呼び出し階層:

⊑(InferenceLattice, a, b)                       [typelattice.jl:395]
  ├─ ignorelimited(a), ignorelimited(b)          [typelattice.jl:389-390]
  ├─ ⊑(widenlattice, ...)                        → 下位層に委譲
  └─ causes ⊆ 判定                               [typelattice.jl:411]

  ⊑(AnyConditionalsLattice, a, b)                [typelattice.jl:414]
    ├─ issubconditional(lattice, a, b)            [typelattice.jl:281-296]
    │   └─ is_same_conditionals(a, b)             [typelattice.jl:298-299]
    └─ ⊑(widenlattice, ...)                       → 下位層に委譲

    ⊑(AnyMustAliasesLattice, a, b)               [typelattice.jl:441]
      ├─ issubalias(a, b)                         [typelattice.jl:381-384]
      ├─ widenmustalias(a)                        [typelattice.jl:314-321]
      └─ ⊑(widenlattice, ...)                     → 下位層に委譲

      ⊑(PartialsLattice, a, b)                   [typelattice.jl:454]
        ├─ フィールドごとに ⊑ 再帰判定
        └─ ⊑(widenlattice, ...)                   → 下位層に委譲

        ⊑(ConstsLattice, a, b)                   [typelattice.jl:532]
          ├─ a.val === b.val (Const同士)
          └─ ⊑(widenlattice, ...)                 → JLTypeLattice

          ⊑(JLTypeLattice, a, b) = a <: b        [abstractlattice.jl:153]

tmerge(InferenceLattice, a, b)                    [typelimits.jl:491]
  └─ tmerge_limited(lattice, a, b)                [typelimits.jl:440]
      ├─ causes の union/merge                    [typelimits.jl:417-438]
      └─ tmerge(widenlattice, ...)                → 下位層に委譲

  tmerge(ConditionalsLattice, a, b)               [typelimits.jl:499]
    ├─ Conditional + Const(true/false) の変換
    ├─ is_same_conditionals → thentype/elsetype マージ
    └─ widenconditional → tmerge(widenlattice, ...)

    tmerge(PartialsLattice, a, b)                 [typelimits.jl:678]
      ├─ tmerge_fast_path                         [typelimits.jl:403]
      ├─ tmerge_partial_struct                    [typelimits.jl:593]
      └─ tmerge(widenlattice, ...)

      tmerge(ConstsLattice, a, b)                 [typelimits.jl:724]
        └─ tmerge(widenlattice, ...)

        tmerge(JLTypeLattice, a, b)               [typelimits.jl:736]
          ├─ Union{a,b} (具象型同士)
          └─ tmerge_types_slow                    [typelimits.jl:779]
              ├─ tname_intersect                  [typelimits.jl:752]
              └─ tuplemerge                       [typelimits.jl:896]

limit_type_size(t, compare, source, depth, len)   [typelimits.jl:17]
  ├─ type_more_complex(t, compare, ...)           [typelimits.jl:223]
  │   └─ is_derived_type_from_any                 [typelimits.jl:77]
  │       └─ is_derived_type                      [typelimits.jl:37]
  └─ _limit_type_size(t, compare, ...)            [typelimits.jl:86]
      └─ __limit_type_size                        [typelimits.jl:192]
```

### データフロー図

```
[抽象解釈エンジン]                         [型格子モジュール]

条件分岐 (isa/===)
  │
  ├─ Conditional(slot, ssadef,      ──▶  typelattice.jl: Conditional 構造体
  │   thentype, elsetype)                  ├─ assert_nested_slotwrapper
  │                                        └─ may_form_limited_typ
  │
  ├─ MustAlias(slot, ssadef,        ──▶  typelattice.jl: MustAlias 構造体
  │   vartyp, fldidx, fldtyp)              ├─ assert_nested_slotwrapper
  │                                        └─ may_form_limited_typ
  │
  ├─ PartialStruct(𝕃, typ,         ──▶  typelattice.jl: PartialStruct (Core定義)
  │   undefs, fields)                      └─ assert_nested_slotwrapper (各field)
  │
  ├─ LimitedAccuracy(typ, causes)   ──▶  typelattice.jl: LimitedAccuracy 構造体
  │
  ▼
型状態のマージ (制御フロー合流点)
  │
  ├─ ⊑(lattice, a, b)              ──▶  typelattice.jl: 格子層ディスパッチ
  │   └─ Bool
  │
  ├─ tmerge(lattice, a, b)         ──▶  typelimits.jl: 格子層ディスパッチ
  │   └─ 合流後の格子要素                   ├─ tmerge_fast_path
  │                                        ├─ tmerge_partial_struct
  │                                        └─ tmerge_types_slow
  │
  ├─ tmeet(lattice, v, t)          ──▶  typelattice.jl: 格子層ディスパッチ
  │   └─ 交差後の格子要素                   └─ typeintersect (JLTypeLattice)
  │
  ├─ limit_type_size(t, c, s, d, l) ──▶ typelimits.jl: 型複雑度制限
  │   └─ 制限後のType                       ├─ type_more_complex
  │                                        └─ _limit_type_size
  │
  ▼
[VarTable (変数型状態テーブル)]
  │
  ├─ stupdate!(lattice, state,      ──▶  typelattice.jl: テーブル全体更新
  │   changes, join_pc)                    └─ smerge → tmerge
  │
  ├─ stoverwrite1!(state, update)   ──▶  typelattice.jl: 単一スロット上書き
  │
  └─ strefine1!(state, refinement)  ──▶  typelattice.jl: 単一スロット精密化
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| typelattice.jl | `Compiler/src/typelattice.jl` | ソース | 格子要素の構造体定義、⊑/tmeet/widenconst、状態管理（smerge/stupdate!/stoverwrite!） |
| typelimits.jl | `Compiler/src/typelimits.jl` | ソース | tmerge 演算、型複雑度制限（limit_type_size/type_more_complex）、issimpleenoughtype |
| abstractlattice.jl | `Compiler/src/abstractlattice.jl` | ソース | 格子層の構造体定義（JLTypeLattice〜InferenceLattice）、⊑/tmerge/tmeet インターフェース定義、fallback_lattice |
| types.jl | `Compiler/src/types.jl` | ソース | AbstractLattice 抽象型の定義 |
| abstractinterpretation.jl | `Compiler/src/abstractinterpretation.jl` | ソース | 格子要素を使用する抽象解釈エンジン |
| boot.jl (Core) | Core モジュール内 | ソース | Const / InterConditional / PartialStruct の定義（グローバルキャッシュ互換のため Core で定義） |
